//
// 8MemPool.h: Memory Buffer Pool Template
//      Powered By 1CoinClear
//      Ref. Ghazmian.Ho GHBLOCK.h, and MFC AfxTempl.h
//  
// * Thread-safe with multiple processor.
// * Don't use TryEnterCriticalSecton() instead of EnterCriticalSecion().
// * Not tested about CACHE LINES with multiple processor.
// * HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\Session Manager\CriticalSectionTimeout
//   defalt value is 2592000 seconds and it is about 90days..


#ifndef __8MEMPOOL_H__
#define __8MEMPOOL_H__


//  Template
template <class TYPE>
class C8MemPool
{
protected:
    struct C8List
    {
        C8List* pNext;
        int nIndex;
    };
public:
// Construction
    C8MemPool( int nBlockSize, int nExtraBlockSize=0 );

// Attributes
    int GetCount();
    int GetEmpty();
    int GetIndex( TYPE *ptr );

// Operations
    TYPE* Alloc();
    BOOL Free( TYPE *ptr );

// Implementation
protected:
    int m_nCount;
    int m_nBlockSize;
    int m_nExBlockSize;
    int m_nListBlockSize;

    PVOID m_pBlocks;
    C8List* m_pFreeList;

    CRITICAL_SECTION m_cs;

public:
    ~C8MemPool();
};

//  ------------------------------------------------------------------------------------------------
//  Constructor
template <class TYPE>
C8MemPool<TYPE>::C8MemPool<TYPE>( int nBlockSize, int nExtraBlockSize )
{
	assert( nBlockSize > 0 );

    m_nCount = 0;
    m_pFreeList = NULL;
    m_pBlocks = NULL;

    m_nBlockSize = nBlockSize;
    m_nExBlockSize = nExtraBlockSize;
    m_nListBlockSize = sizeof(C8List) + sizeof(TYPE) + m_nExBlockSize;

    m_pBlocks =
        VirtualAlloc(
            NULL, 
            m_nBlockSize * m_nListBlockSize,
            MEM_RESERVE | MEM_COMMIT,           // reserve and commit
            PAGE_READWRITE 
        );

	assert( m_pBlocks );

    C8List* pList = (C8List*) m_pBlocks;

	assert( pList );

    // init linear linked list for buffer pool
    pList = (C8List*) ((DWORD)pList + (m_nBlockSize - 1) * m_nListBlockSize);
    for( int i = m_nBlockSize-1; i >= 0; i-- )
    {
        pList->pNext = m_pFreeList;
        pList->nIndex = i;
        m_pFreeList = pList;
#ifdef _DEBUG
        if( m_nExBlockSize )
            memset( (PVOID)((DWORD)pList+sizeof(C8List)+sizeof(TYPE)),
            (i%10+'0'), m_nExBlockSize );
#endif
        pList = (C8List*)((DWORD)pList - m_nListBlockSize);
    }

#if 0
    InitializeCriticalSection( &m_cs );
#else
    InitializeCriticalSectionAndSpinCount( &m_cs, 4000 );   // why 4000?
                                                            // by Jeffrey Richter
#endif
                                                            // dwSpinCount = [0..0x00FFFFFF]
                                                            // single processor machine will ignore this value and set to 0
}

template <class TYPE>
C8MemPool<TYPE>::~C8MemPool<TYPE>()
{
    if( NULL != m_pBlocks )
        VirtualFree( m_pBlocks, 0, MEM_RELEASE );

    DeleteCriticalSection( &m_cs );
}

template <class TYPE>
TYPE* C8MemPool<TYPE>::Alloc()
{
    C8List* pList = NULL;
    TYPE* pType = NULL;

    EnterCriticalSection( &m_cs );
//  {
        pList = m_pFreeList;
        if( pList != NULL )
        {
            m_pFreeList = m_pFreeList->pNext;
            m_nCount++;
            pType = (TYPE*)(pList+1);
			assert( m_nCount > 0 );  // make sure we don't overflow
        }
//  }
    LeaveCriticalSection( &m_cs );

    return pType;
}

template <class TYPE>
BOOL C8MemPool<TYPE>::Free( TYPE *ptr )
{
    BOOL bRet = FALSE;

    EnterCriticalSection( &m_cs );
//  {
        C8List* pList = ((C8List*)(ptr))-1;
        if( m_nCount > 0 )
        {
            pList->pNext = m_pFreeList;
            m_pFreeList = pList;
            m_nCount--;
            bRet = TRUE;
        }
//  }
    LeaveCriticalSection( &m_cs );

    return bRet;
}

template <class TYPE>
int C8MemPool<TYPE>::GetCount()
{
    int nRet = 0;

    EnterCriticalSection( &m_cs );
//  {
        nRet = m_nCount;
//  }
    LeaveCriticalSection( &m_cs );

    return nRet;
}

template <class TYPE>
int C8MemPool<TYPE>::GetEmpty()
{
    int nRet = 0;

    EnterCriticalSection( &m_cs );
//  {
        nRet = m_nBlockSize - m_nCount;
//  }
    LeaveCriticalSection( &m_cs );

    return nRet;
}

template <class TYPE>
int C8MemPool<TYPE>::GetIndex( TYPE *ptr )
{
    int nRet = 0;

    EnterCriticalSection( &m_cs );
//  {
        C8List* pList = ((C8List*)(ptr))-1;
        nRet = pList->nIndex;
//  }
    LeaveCriticalSection( &m_cs );

    return nRet;
}

#endif